;
;==================================================================================================
;   SCSI2IDE
;   WAYNE WARTHEN - 2013-01-24
;==================================================================================================
;
; WILL BOOT TO ZAPPLE PROMPT.
; USE ZAPPLE 'G' (GO) COMMAND TO START FUNCTIONS:
;   G1000 = SCSI TARGET MODE PROCESSOR (PRESS A KEY TO EXIT)
;   G1010 = ZERO FIRST 256 BLOCKS OF IDE DEVICE
;   G1020 = READ BLOCK OF STORAGE INTO BUFFER
;   G1030 = WRITE BLOCK OF STORAGE FROM BUFFER
;   G1040 = DUMP CURRENT STORAGE BUFFER
;
; NOTES:
;   1) TARGET MODE OPERATION ONLY
;   2) INTENT IS SCSI-1/2 PROTOCOL & COMPLIANT WITH SASI
;
; TODO:
;   X) SET PHASE SIGNALS FIRST, THEN START DRIVING DATA, BUT PRIOR TO LOOPING
;   X) BE SURE TO STOP DRIVING DATA BEFORE DROPPING I/O
;   X) MAINTAIN SENSE BYTE DYNAMICALLY AND RETURN PROPER VALUE
;   X) CHECK FOR ATN AT START OF ANY PHASE
;   X) SEND CHECK CONDITION (SENSE CODE ILLEGAL REQUEST) FOR INVALID LUN FOR ALL COMMANDS, BUT:
;      A) IF COMMAND IS REQUEST SENSE, RETURN CURRENT SENSE BYTE
;      B) IF COMMAND IS INQUIRY, RETURN WITH FIRST BYTE=$7F
;   6) MAINTAIN LUN-SPECIFIC SENSE CODES (NO DISCONNECT, SO MAY NOT BE NEEDED)
;   X) IMPLEMENT PSEUDO-DMA
;   X) IMPLEMENT CHIP-BASED TARGET SELECTION
;   7) DETECT BUS RESET VIA INTERRUPT
;   X) SUPPORT MULTIPLE LUNS
;   X) CONTROL DEBUG TRACING VIA CONDITIONAL
;   XX) REFACTOR PPIDE READ/WRITE FOR PERFORMANCE
;   XX) ENSURE DELAYS ARE CORRECT (CABLE DESKEW AND BUS SETTLE)
;   XX) RESPECT BOARD CONFIG SWITCH SETTINGS
;   XX) AUTOSTART TARGET MODE COMMAND PROCESSOR
;   XX) IMPLEMENT FORMAT COMMAND
;   XX) IMPLEMENT VERIFY COMMAND
;   XX) DETECT DRIVE READ/WRITE ERRORS AND PROPAGATE TO SCSI RESPONSE
;   XX) OPTIMIZE READ/WRITE LOOPS WITH Z80 INI/OUTI
;   18) IMPLEMENT PASS-THRU MODE WHERE FULL IDE/SD DEVICE IS PRESENTED AS A SINGLE LUN
;   XX) FIX DISK BUFFER ADDRESS
;
;==================================================================================================
;   CONFIGURATION BYTE DEFINITION
;==================================================================================================
; +-----------------+
; | 7 6 5 4 3 2 1 0 | JUMPER BLOCK P1
; +-----------------+
;
;     P1
;    +---+
;  0 | D | --\ 
;  1 | D |    +-- SCSI DEVICE ID (0-7)
;  2 | D | --/
;  3 | B | ------ BLOCK SIZE (0=512, 1=256)
;  4 | X | ------ UNUSED
;  5 | W | ------ WP (0=OFF, 1=ON)
;  6 | G | ------ DEBUG (0=OFF, 1=ON)
;  7 | T | ------ STORAGE DEVICE (0=IDE, 1=SD)
;    +---+
;
; NOTE THAT BIT 4 WAS DEDICATED TO DRQ IN THE SECOND RUN OF THE PROTOTYPE BOARDS AND IS
; NOT AVAILABLE FOR THIS BOARD VARIANT.
;
;==================================================================================================
;   MEMORY MAP
;==================================================================================================
;
; STANDARD Z80 ADDRESS SPACE IS 64K
; BOTTOM HALF (32K) MAPPED TO ROM, TOP HALF MAPPED TO RAM
;
; ROM - $0000-$7FFF (LOWER 32K)
; -----------------------------
; $0000-$00FF	Z80 PAGE ZERO
; $0100-$0FFF	SYSTEM INITIALIZATION
; $1000-$5FFF	SCSI CONTROL PROGRAM
; $6000-$6FFF	ZAPPLE MONITOR
; $7000-$7FFF	HBIOS (RELOCATED TO $F000-$FFFF AT STARTUP)
;
; RAM - $8000-$FFFF (UPPER 32K)
; -----------------------------
; $8000-$EDFF	WORKING DATA STORAGE FOR SCSI CONTROL PROGRAM
; $EE00-$EFFF	ZAPPLE MONITOR WORKING DATA STORAGE
; $F000-$FFFF	HBIOS (RUNS HERE, CODE AND DATA, STACK SPACE AT TOP)
;
;==================================================================================================
;   PORT DEFINITIONS
;==================================================================================================
;
; I/O BASE NORMALLY 0x80
;
; ---------------------------------
; SCSI CONTROLLER (I/0 BASE + 0x08)
; ---------------------------------
; PORT	READ				WRITE
; ----	------------------------------ 	------------------------------
; 0x88  CURRENT SCSI DATA		OUTPUT DATA REGISTER
; 0x89	INITIATOR COMMAND REGISTER	INITIATOR COMMAND REGISTER
; 0x8A	MODE REGISTER			MODE REGISTER
; 0x8B	TARGET COMMAND REGISTER		TARGET COMMAND REGISTER
; 0x8C	CURRENT SCSI BUS STATUS		SELECT ENABLE REGISTER
; 0x8D  BUS & STATUS REGISTER		START DMA SEND
; 0x8E	INPUT DATA REGISTER		START DMA TARGET RECEIVE
; 0x8F	RESET PARITY/INTERRUPT		START DMA INITIATOR RECEIVE
;
; -----------------------------------------------
; PARALLEL PORT (IDE) INTERFACE (I/O BASE + 0x00)
; -----------------------------------------------
; PORT	PPI				IDE
; ----	------------------------------ 	------------------------------
; 0x80  PORT A				LSB
; 0x81  PORT B				MSB
; 0x82  PORT C				IDE CONTROL
; 0x83  CONTROL
;
; -----------------------------------------------
; CONFIG JUMPER P1 (I/O BASE + 0x04)
; -----------------------------------------------
; BIT(S)	DEFINITION
; ------	------------------------------------------------------
; 0-2		SCSI BUS DEVICE ID
; 3		BLOCK SIZE (0=512, 1=256)
; 4		DEBUG (0=OFF, 1=ON)
; 5-6		WP (WHAT IS THIS?)
; 7		STORAGE DEVICE (0=IDE, 1=SD)
;
; -----------------------------------------------
; SERIAL PORT INTERFACE (I/O BASE + 0x10)
; -----------------------------------------------
; PORT	READ W/ DLAB=0			WRITE W/ DLAB=0			READ/WRITE W/ DLAB=1	
; ----	------------------------------ 	------------------------------ 	------------------------------
; 0x90	RECEIVE BUFFER REGISTER		TRANSMIT HOLDING REGISTER	DIVISOR LATCH (LSB)
; 0x91	INTERRUPT ENABLE REGISTER	INTERRUPT ENABLE REGISTER	DIVISOR LATCH (MSB)
; 0x92	INTERRUPT IDENT REGISTER	FIFO CONTROL			N/A
; 0x93	LINE CONTROL REGISTER		LINE CONTROL REGISTER		N/A
; 0x94	MODEM CONTROL REGISTER		MODEM CONTROL REGISTER		N/A
; 0x95	LINE STATUS REGISTER		LINE STATUS REGISTER		N/A
; 0x96	MODEM STATUS REGISTER		MODEM STATUS REGISTER		N/A
; 0x97  SCRATCH REGISTER		SCRATCH REGISTER		N/A
;
;==================================================================================================
;   DEFINITIONS
;==================================================================================================
;
#INCLUDE "std.asm"
;
#DEFINE		S2I_VERTAG	"0.6a"	; 4 BYTE VERSION TAG STRING
;
; THE FOLLOWING MACRO CAN BE USED TO ADD AN ARBITRARY DELAY BETWEEN PHASES.
; IF DESIRED, UNCOMMENT THE LINE AND SET TO DESIRED DELAY (S2I_DELAY1, S2I_DELAY2,
; S2I_DELAY4, S2I_DELAY8, ..., S2I_DELAY256)
;
;#DEFINE		S2I_IPDELAY	S2I_DELAY4	; INTERPHASE DELAY
;
; SCSI CONTROLLER PORT ADDRESSES
;
S2I_BASE	.EQU	$80

SCSI_BASE	.EQU	S2I_BASE + 8
;
; SCSI CONTROLLER REGISTER PORTS
;
S2I_CDR		.EQU	SCSI_BASE + 0	; CURRENT SCSI DATA REGISTER
S2I_ICR		.EQU	SCSI_BASE + 1	; INITIATOR COMMAND REGISTER
S2I_MR		.EQU	SCSI_BASE + 2	; MODE REGISTER
S2I_TCR		.EQU	SCSI_BASE + 3	; TARGET COMMAND REGISTER
S2I_CBS		.EQU	SCSI_BASE + 4	; CURRENT SCSI BUS STATUS REGISTER
S2I_SER		.EQU	SCSI_BASE + 4	; SELECT ENABLE REGISTER
S2I_BSR		.EQU	SCSI_BASE + 5	; BUS AND STATUS REGISTER
S2I_SDS		.EQU	SCSI_BASE + 5	; START DMA SEND
S2I_IDR		.EQU	SCSI_BASE + 6	; INPUT DATA REGISTER
S2I_SDTR	.EQU	SCSI_BASE + 6	; START DMA TARGET RECEIVE
S2I_RPI		.EQU	SCSI_BASE + 7	; RESET PARITY/INTERRUPT
S2I_SDIR	.EQU	SCSI_BASE + 7	; START DMA INITIATOR RECEIVE
;
; PHASE CONFIGURATION BITS
;
S2I_PHMSGOUT	.EQU	%00000110	; MSG + C/D
S2I_PHMSGIN	.EQU	%00000111	; MSG + C/D + I/O
S2I_PHCMD	.EQU	%00000010	; C/D
S2I_PHSTATUS	.EQU	%00000011	; C/D + I/O
S2I_PHDATAOUT	.EQU	%00000000	; <NOTHING ASSERTED>
S2I_PHDATAIN	.EQU	%00000001	; I/O
;
;==================================================================================================
;   DERIVED VALUES
;==================================================================================================
;
; THE LUN OFFSET IS THE VALUE ADDED TO THE HIGH WORD OF THE 32-BIT BLOCK VALUE
; TO ADJUST FOR THE LUN BEING ADDRESSED.  INCREMEMENTING THE HIGH WORD BY ONE
; IS "WORTH" 32MB.
;
S2I_LUN_OFFSET	.EQU	(S2I_LUNSIZ + 31) / 32
;
;==================================================================================================
;   CONFIGURATION
;==================================================================================================
;
; CONFIGURATION SETTINGS HAVE BEEN MOVED TO "CONFIG_S2I_XXX.ASM" FILE(S)
;
;==================================================================================================
;   PAGE ZERO AND STARTUP CODE
;==================================================================================================
;
	.ORG	$0
	JP	INIT			; COLD START VECTOR
	.FILL	$08-$
	JP	$FF20			; HBIOS DISPATCH
	.FILL	$38-$
	JP	$601E			; ZAPPLE BREAKPOINT VECTOR
	.FILL	$100-$
;
; INITIALIZATION
;
INIT:
	DI
	IM	1
	LD	SP,0			; STACK AT TOP OF MEMORY
;	
	; EMIT FIRST SIGN OF LIFE TO SERIAL PORT
	CALL	XIO_INIT		; INIT SERIAL PORT
	LD	HL,STR_BOOT		; POINT TO MESSAGE
	CALL	XIO_OUTS		; SAY HELLO
	CALL	XIO_DOT			; MARK PROGRESS
;
	; COPY HBIOS TO RAM
	LD	HL,$7000		; HBIOS ROM IMAGE LOC
	LD	DE,$F000		; HBIOS OPERATING LOC
	LD	BC,$1000		; SIZE OF HBIOS
	LDIR				; COPY IT
	CALL	XIO_DOT			; MARK PROGRESS
;
	; INITIALIZE HBIOS
	CALL	$7000			; CALL HBIOS INIT AT $7000
;
	; PASS CONTROL TO ZAPPLE
	JP	$6000			; GO TO ZAPPLE
;
STR_BOOT	.DB	"Boot$"
;
; IMBED DIRECT SERIAL I/O ROUTINES
;
#INCLUDE "xio.asm"
;
;	ROM AREA FROM $1000 TO $5FFF IS USED FOR S2I CONTROL PROGRAM
;
	.FILL	$1000 - $
;
;==================================================================================================
;   FUNCTION DISPATCHING
;==================================================================================================
;
	.FILL	$1000 - $,$FF
	; SCSI TARGET MODE OPERATION (PRESS A KEY TO EXIT)
	CALL	S2I_DOIT		; CALL FUNCTION EXECUTOR
	.DW	S2I_RUN			; TOP OF STACK POINTS TO FUNC ADDRESS
;
	.FILL	$1010 - $,$FF
	; ZERO THE IDE DEVICE (FIRST 256 BLOCKS)
	CALL	S2I_DOIT		; CALL FUNCTION EXECUTOR
	.DW	S2I_ZERO		; TOP OF STACK POINTS TO FUNC ADDRESS
;
	.FILL	$1020 - $,$FF
	; READ A BLOCK OF DATA INTO BUFFER
	CALL	S2I_DOIT		; CALL FUNCTION EXECUTOR
	.DW	S2I_RDBLK		; TOP OF STACK POINTS TO FUNC ADDRESS
;
	.FILL	$1030 - $,$FF
	; WRITE A BLOCK OF DATA INTO BUFFER
	CALL	S2I_DOIT		; CALL FUNCTION EXECUTOR
	.DW	S2I_WRBLK		; TOP OF STACK POINTS TO FUNC ADDRESS
;
	.FILL	$1040 - $,$FF
	; DISPLAY BUFFER DATA
	CALL	S2I_DOIT		; CALL FUNCTION EXECUTOR
	.DW	S2I_DUMP		; TOP OF STACK POINTS TO FUNC ADDRESS
;
S2I_DOIT:
;
	; BASIC INITIALIZATION
	DI				; NO INTERRUPTS PLEASE
	IM	1			; INTERRUPT MODE 1
;
	; GET ROUTINE ADDRESS POINTED TO BY TOP OF STACK
	POP	HL			; GET THE FUNCTION ADDRESS POINTER
	LD	A,(HL)			; DEREFERENCE
	INC	HL			; "
	LD	H,(HL)			; "
	LD	L,A			; "

	; SETUP USER STACK AT TOP OF MEMORY
	LD	SP,0			; STACK AT TOP OF MEMORY
;
	; RUN THE FUNCTION
	CALL	JPHL			; CALL FUNCTION IN HL
;
	; RETURN TO MONITOR
	CALL	NEWLINE			; SPACER
	CALL	NEWLINE			; SPACER
	RST	38			; INVOKE MONITOR
;
;
;==================================================================================================
;   CONTROL PROGRAM STARTS HERE
;==================================================================================================
;
S2I_RUN:
;
	LD	DE,STR_BANNER		; POINT TO BANNER
	CALL	WRITESTR		; DISPLAY IT
;
	IN	A,($84)			; READ CONFIG BYTE
	IN	A,($84)			; ... TWICE TO HANDLE OLDER HARDWARE
	LD	(S2I_CONFIG),A		; SAVE IT
;
	LD	DE,STR_CONFIG		; POINT TO CONFIG BYTE ANNONCEMENT
	CALL	WRITESTR		; DISPLAY
	LD	A,(S2I_CONFIG)		; LOAD CONFIG BYTE VALUE
	CALL	PRTHEXBYTE		; DISPLAY IT
;
	LD	DE,STR_DBGLVL		; DEBUG LEVEL PREFIX STRING
	CALL	WRITESTR		; DISPLAY
	LD	A,(S2I_CONFIG)		; LOAD CONFIG BYTE VALUE
	RLCA				; ROTATE BEBUG BIT TO BIT 0
	RLCA				; ...
	AND	%00000001		; ISOLATE IT
	INC	A			; INCREMENT SO IT IS EITHER 1 OR 2
	LD	(S2I_DBGLVL),A		; SAVE IT
	CALL	PRTHEXBYTE		; DISPLAY IT
;
	LD	A,(S2I_CONFIG)		; LOAD CONFIG BYTE VALUE
	AND	%00000111		; ISOLATE SCSI DEVICE ID BITS
	LD	(S2I_SID),A		; SAVE SCSI ID
	LD	A,(S2I_CONFIG)		; LOAD CONFIG BYTE VALUE
	LD	B,A			; MOVE TO B
	LD	A,2			; ASSUME 512 BYTE BLOCK SIZE
	BIT	3,B			; SAMPLE BIT 3
	JR	Z,S2I_RUN0		; YES, DO IT
	LD	A,1			; OTHERWISE 256 BYTE BLOCK SIZE
S2I_RUN0:
	LD	(S2I_BLKSIZE),A		; SAVE BLOCKSIZE
;
	LD	A,(S2I_CONFIG)		; LOAD CONFIG BYTE VALUE
	LD	B,A			; MOVE TO B
	LD	A,DIODEV_PPIDE		; ASSUME IDE DEVICE
	BIT	7,B			; SAMPLE DEVICE SELECTION BIT
	JR	Z,S2I_RUN00		; IF 0, IT IS IDE, SKIP AHEAD
	LD	A,DIODEV_SD		; MUST BE SD DEVICE
S2I_RUN00:
	LD	(S2I_DEVICE),A		; SAVE DEVICE SELECTION
;
	LD	DE,STR_XFRMODE		; POINT TO TRANSFER MODE ANNOUNCEMENT
	CALL	WRITESTR		; DISPLAY
;
	LD	DE,STR_DEVIDE		; POINT TO IDE DEVICE ANNOUNCEMENT
	LD	A,(S2I_DEVICE)		; GET DEVICE ID
	CP	DIODEV_PPIDE		; IDE?
	JR	Z,S2I_RUN000		; DISPLAY IT
	LD	DE,STR_DEVSD		; POINT TO SD DEVICE ANNUNCEMENT
	CP	DIODEV_SD		; SD?
	JR	Z,S2I_RUN000		; DISPLAY IT
	LD	DE,STR_DEVUNK		; UNKNOWN
S2I_RUN000:
	CALL	WRITESTR		; DISPLAY
;
	LD	DE,STR_CONFIG1		; POINT TO CONFIG ANNOUNCEMENT, PART 1
	CALL	WRITESTR		; DISPLAY IT
	LD	A,(S2I_SID)		; LOAD THE SCSI ID
	CALL	PRTHEXBYTE		; DISPLAY IT
	LD	DE,STR_CONFIG2		; POINT TO CONFIG ANNOUNCEMENT, PART 2
	CALL	WRITESTR		; DISPLAY IT
	LD	A,(S2I_BLKSIZE)		; GET BLOCKSIZE
	LD	DE,STR_BLK512		; ASSUME 512
	CP	2			; CHECK FOR 512 (VALUE = 2)
	JR	Z,S2I_RUN1		; IF SO, GO TO DISPLAY
	LD	DE,STR_BLK256		; ASSUME 256
	CP	1			; CHECK FOR 256 (VALUE = 1)
	JR	Z,S2I_RUN1		; IF SO, GO TO DISPLAY
	LD	DE,STR_BLKUNK		; ASSUME UNKNOWN
	JR	Z,S2I_RUN1		; GO TO DISPLAY
S2I_RUN1:
	CALL	WRITESTR		; DISPLAY BLOCK SIZE
	LD	DE,STR_CONFIG3		; POINT TO CONFIG ANNOUNCEMENT, PART 3
	CALL	WRITESTR		; DISPLAY IT
	LD	A,S2I_LUNCNT		; LOAD THE LUN COUNT
	CALL	PRTHEXBYTE		; DISPLAY IT
	LD	DE,STR_CONFIG4		; POINT TO CONFIG ANNOUNCEMENT, PART 4
	CALL	WRITESTR		; DISPLAY IT
	LD	HL,S2I_LUNSIZ
	CALL	PRTDEC			; DISPLAY IT
	LD	DE,STR_CONFIG5		; POINT TO CONFIG ANNOUNCEMENT, PART 5
	CALL	WRITESTR		; DISPLAY IT
;
	; SET DISK BUFFER ADDRESS
	LD	HL,S2I_DSKBUF		; ADDRESS OF DEDICATED DISK BUFFER
	LD	B,BF_DIOSETBUF		; HBIOS SET BUF FUNCTION
	RST	08			; DO IT
;
	; PRE-LOOP INITIALIZATION
	XOR	A			; A := 0
	LD	(S2I_SENSE),A		; INITIALIZE SENSE
;
;===============================================================
; SCSI TARGET MODE PROCESSING LOOP
;===============================================================
;
S2I_TGTPROC:
	; INITIALIZATION
	XOR	A			; A = 0
	LD	(S2I_STAT),A		; INITIALIZE STATUS
	DEC	A			; A =$FF
	LD	(S2I_CBS_PRV),A		; RESET SAVED CBS
	LD	(S2I_BSR_PRV),A		; RESET SAVED BSR
	LD	(S2I_LUN),A		; INITIALIZE ACTIVE LUN ($FF SIGNIFIES INVALID)
;
	CALL	S2I_INITTGT		; INITIALIZE FOR TARGET MODE
#IF (S2I_TRACE >= 2)
	LD	DE,STR_BUSFREE		; ANNOUNCE THE BUS IS FREE
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
	CALL	S2I_SELECT		; WAIT FOR SELECTION OR KEY
	JR	NZ,S2I_TGTPROC1		; NZ = KEY PRESSED, ABORT
	CALL	S2I_CMDPROC		; GET AND HANDLE COMMAND
;
	; CLEAR SENSE BYTE IF COMMAND WAS SUCCESSFUL
	LD	A,(S2I_STAT)		; GET FINAL STATUS
	OR	A			; SET FLAGS
	JR	NZ,S2I_TGTPROC		; BAD STATUS, LEAVE SENSE VALUE
	XOR	A			; GOOD STATUS, A := 0
	LD	(S2I_SENSE),A		; SENSE := 0
	JR	S2I_TGTPROC		; LOOP
;	
S2I_TGTPROC1:	; HANDLE USER KEYSTROKE
;
	CP	$1B			; <ESC>?
	JR	NZ,S2I_TGTPROC		; NOPE, RESTART TGT CMD PROCESSOR
;
	; <ESC> PRESSED, CLEAN UP AND RETURN
	CALL	S2I_INITTGT		; FREE BUS
;
	LD	DE,STR_DONE		; ANNOUNCE WE ARE DONE
	CALL	WRITESTR		; PRINT IT
;	
	RET				; DONE
;
;===============================================================
; SCSI TARGET MODE INITIALIZATION
;===============================================================
;
S2I_INITTGT:
	LD	A,$40			; MODE: TARGET
	OUT	(S2I_MR),A		; SET MODE REGISTER
;
	; CLEAR ALL SIGNALS OFF THE BUS
	XOR	A
	OUT	(S2I_ICR),A		; CLEAR INITIATOR COMMAND REGISTER
	OUT	(S2I_TCR),A		; CLEAR TARGET COMMAND REGISTER
	OUT	(S2I_CDR),A		; ZEROES TO OUTPUT DATA REGISTER
	OUT	(S2I_SER),A		; ZEROES TO SELECT ENABLE REGISTER
	RET				; DONE

#IF 1
;
;===============================================================
; SCSI SELECTION
;===============================================================
;
; SCSI TARGET SELECTION
; WAIT FOR THE INITIATOR TO SELECT US
; AND ACCEPT SELECT REQUEST BY SETTING BSY
;
S2I_SELECT:
#IF (S2I_TRACE >= 2)
	LD	DE,STR_SELECT		; POINT TO ANNOUNCEMENT
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
	; SET BIT IN A FOR OUR SCSI ID
	LD	A,(S2I_SID)		; GET OUR SCSI DEVICE ID
	LD	B,A			; B := OUR SCSI ID
	INC	B			; WE WANT TO LOOP ONE EXTRA TIME
	XOR	A			; A := 0
	SCF				; SET CARRY FLAG
;
S2I_SELECT0:	; LOOP TO ROTATE CF BIT INTO DESIRED SELECT BIT POSITION
	RLA				; ROLATE LEFT BY ONE BIT
	DJNZ	S2I_SELECT0		; LOOP UNTIL DONE
	LD	(S2I_SIDMASK),A		; SAVE DEVICE ID MASK
	OUT	(S2I_SER),A		; SETUP SELECT ENABLE REGISTER
;
S2I_SELECT1:	; ENTRY POINT FOR LOOPING, RESET INT THEN PROCEED
	IN	A,(S2I_RPI)		; RESET INTERRUPT
;
S2I_SELECT2:	; SELECT LOOP
;
	; CHECK FOR A USER KEYRESS WHICH MEANS ESCAPE TO MONITOR
	CALL	S2I_KEYCHK		; KEY PRESSED?
	OR	A			; SET FLAGS
	RET	NZ			; RETURN IF KEY PRESSED
;
	; CHECK FOR INT REQ WHICH MEANS A SELECT HAS OCCURRED
	IN	A,(S2I_BSR)		; GET STATUS
	BIT	4,A			; CHECK BIT 4, INT REQ
	JR	Z,S2I_SELECT2		; NOPE, LOOP
;
#IF (S2I_TRACE >= 2)
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,S2I_SNAP		; DUMP A SNAPSHOT OF REGISTERS!
#ENDIF
;
	; INT REQ IS AMBIGUOUS HERE!!! COULD ALSO MEAN A BUS RESET
	; HAS OCCURRED.  BELOW IS A CHECK TO DETERMINE IF THIS
	; WAS ACTUALLY A SELECT.  IF NOT, WE ASSUME A BUS RESET
	; MUST HAVE BEEN TRIGGERED AND RESTART FROM SCRATCH!
	IN	A,(S2I_CBS)		; GET CURRENT BUS STATUS
	BIT	1,A			; ISOLATE BIT 1
	JR	Z,S2I_SELECT		; NOT SET, BUS RESET? DO TOTAL RESTART
;;
;	; CHECK FOR BUS RESET
;	IN	A,(S2I_CBS)		; GET STATUS
;	BIT	7,A			; CHECK BIT 4, INT REQ
;	JR	NZ,S2I_SELECT		; YIKES, BUS RESET, TOTAL RESTART!
;
	; ENSURE THAT NO MORE THAN 2 DATA BITS ARE SET (INITIATOR AND TARGET ID)
	IN	A,(S2I_CDR)		; GET THE CURRENT DATA REGISTER
	LD	B,8			; INITIALIZE LOOP COUNT
	LD	E,0			; INITIALIZE BIT COUNT
S2I_SELECT4:	
	RRA				; NEXT BIT INTO CF
	JR	NC,S2I_SELECT5		; BIT NOT SET, SKIP INCREMENT
	INC	E			; COUNT A BIT THAT IS SET
S2I_SELECT5:
	DJNZ	S2I_SELECT4		; DO ALL 8 BITS
	LD	A,E			; GET THE BIT COUNT
	CP	3			; 3 OR MORE???
	JR	NC,S2I_SELECT1		; TOO MANY DATA BITS SET, KEEP TRYING
;
	; BOGOSITY FAILSAFE: CHECK THAT OUR DATA BIT IS STILL ASSERTED
	IN	A,(S2I_CDR)		; GET THE CURRENT DATA REGISTER
	LD	HL,S2I_SIDMASK		; POINT TO OUR SCSI ID MASK
	AND	(HL)			; APPLY MASK
	JR	Z,S2I_SELECT1		; OOPS, NO LONGER SET, LOOP
;
	; OK, WE ARE BEING SELECTED, SET BSY
	IN	A,(S2I_ICR)		; GET CURRENT ICR
	OR	%00001000		; ASSERT BSY
	OUT	(S2I_ICR),A		; OUTPUT
;
S2I_SELECT6:	; WAIT FOR INITIATOR TO DEASSERT SEL
	IN	A,(S2I_CBS)		; GET BUS STATUS
	AND	%00000010		; ISOLATE SEL
	JR	NZ,S2I_SELECT6		; STILL SET, LOOP
;
	; CLEAR THE SELECT ENABLE REGISTER & CLEAR INTERRUPTS
	XOR	A			; A := 0
	OUT	(S2I_SER),A		; CLEAR SELECT ENABLE REGISTER
	IN	A,(S2I_RPI)		; RESET INTERRUPT???
;	
	; SELECTION COMPLETE!
	XOR	A			; A=0
	RET				; DONE
#ENDIF
#IF 0
;
;===============================================================
; SCSI SELECTION
;===============================================================
;
; SCSI TARGET SELECTION
; WAIT FOR THE INITIATOR TO SELECT US
; AND ACCEPT SELECT REQUEST BY SETTING BSY
;
S2I_SELECT:
#IF (S2I_TRACE >= 2)
	LD	DE,STR_SELECT
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
S2I_SELECT0:
	CALL	S2I_KEYCHK		; KEY PRESSED?
	OR	A			; SET FLAGS
	RET	NZ			; RETURN IF KEY PRESSED
;
	; GET CURRENT BUS STATUS
	IN	A,(S2I_CBS)		; GET CURRENT BUS STATUS
	LD	B,A			; SAVE IT IN B
	IN	A,(S2I_BSR)		; GET BUS STATUS
	LD	C,A			; SAVE IT IN C
	IN	A,(S2I_CDR)		; GET CURRENT DATA VALUE
	LD	D,A			; SAVE IT IN D
;
	; SEE IF BUS STATE CHANGED
	LD	A,(S2I_CBS_PRV)		; GET PREVIOUS CBS
	CP	B			; COMPARE TO CURRENT
	JR	NZ,S2I_SELECT1		; IF CHANGE, PRINT
	LD	A,(S2I_BSR_PRV)		; GET PREVIOUS BSR
	CP	C			; COMPARE TO CURRENT
	JR	NZ,S2I_SELECT1		; IF CHANGE, PRINT
	JR	S2I_SELECT2		; NO CHANGE BYPASS PRINTING
;
S2I_SELECT1:	; PRINT CURRENT BUS STATUS
	LD	A,B			; RECOVER CBS
	LD	(S2I_CBS_PRV),A		; SAVE IT AS PREVIOUS VALUE
	CALL	PRTHEXBYTE		; PRINT IT
	CALL	PC_COLON		; SEPARATOR
	LD	A,C			; RECOVER BSR
	LD	(S2I_BSR_PRV),A		; SAVE IT AS PREVIOUS VALUE
	CALL	PRTHEXBYTE		; PRINT IT
	CALL	PC_COLON		; SEPARATOR
	LD	A,D			; RECOVER CDR
	CALL	PRTHEXBYTE		; PRINT IT
	CALL	PC_SPACE		; SEPARATOR
;
S2I_SELECT2:	; TEST FOR VALID SELECT PHASE
	LD	A,B			; GET BUS STATUS
	AND	%11100110		; BITS RES, BSY, REQ, MSG, C/D, I/O, SEL, DBP
	CP	%00000010		;      0,   0,   0,   X,   X,   0,   1,   X
	JR	NZ,S2I_SELECT0		; NOPE, LOOP
;
S2I_SELECT3:	; SELECT ACTIVE, TEST FOR OUR ID
	LD	A,D			; NOW GET CURRENT DATA VALUE
	BIT	0,A			; IS OUR ID BIT SET?
	JR	Z,S2I_SELECT0		; NOPE, LOOP
;
	; ENSURE THAT NO MORE THAN 2 DATA BITS TO BE SET (INITIATOR AND TARGET ID)
	LD	A,D			; MAKE SURE WE HAVE CURRENT DATA VALUE
	LD	B,8			; INITIALIZE LOOP COUNT
	LD	C,0			; INITIALIZE BIT COUNT
S2I_SELECT4:	
	RRA				; NEXT BIT INTO CF
	JR	NC,S2I_SELECT5		; BIT NOT SET, SKIP INCREMENT
	INC	E			; COUNT A BIT THAT IS SET
S2I_SELECT5:
	DJNZ	S2I_SELECT4		; DO ALL 8 BITS
	LD	A,E			; GET THE BIT COUNT
	CP	3			; 3 OR MORE???
	JR	NC,S2I_SELECT0		; TOO MANY DATA BITS SET, KEEP TRYING
;
	; OK, WE ARE BEING SELECTED, SET BSY
	IN	A,(S2I_ICR)		; GET CURRENT ICR
	OR	%00001000		; ASSERT BSY
	OUT	(S2I_ICR),A		; OUTPUT
;
S2I_SELECT6:	; WAIT FOR INITIATOR TO DEASSERT SEL
	IN	A,(S2I_CBS)		; GET BUS STATUS
	AND	%00000010		; ISOLATE SEL
	JR	NZ,S2I_SELECT6		; STILL SET, LOOP
;
	; SELECTION PHASE COMPLETE!
	XOR	A			; A=0
	RET				; DONE
#ENDIF
;
;===============================================================
; SCSI MESSAGE PROCESSOR
;===============================================================
;
; CHECK FOR INITIATOR MESSAGES AND HANDLE AS APPROPRIATE
;
S2I_MSGPROC:	; CHECK FOR INITIATOR MESSAGE PENDING (ATN ASSERTED)
;
	PUSH	AF
S2I_MSGPROCX1:
	IN	A,(S2I_BSR)		; GET BSR VALUE
	AND	%00000010		; ISOLATE ATN
	JR	Z,S2I_MSGPROCX2		; NOTHING PENDING, DONE
	CALL	S2I_MSGPROC3		; ATTENTION ACTIVE, HANDLE IT
	JR	S2I_MSGPROCX1		; LOOP AS NEEDED
;
S2I_MSGPROCX2:
	POP	AF
	RET
;
S2I_MSGPROC3:	; HANDLE ACTIVE ATTENTION
	LD	A,(S2I_MSG)		; GET EXISTING MESSAGE BYTE
	PUSH	AF			; SAVE IT
;
	PUSH	BC			; SAVE ALL OTHER REGISTERS
	PUSH	DE
	PUSH	HL
;
	; GET THE MESSAGE
	CALL	S2I_MSGOUT		; RUN MESSAGE OUT PHASE
;
	; PREP STACK SO THAT ALL HANDLERS RETURN TO MESSAGE PROCESSOR
	LD	HL,S2I_MSGPROC4		; ADDRESS FOR RETURN FROM HANDLER
	PUSH	HL			; PUSH IT ON STACK
;
	; HANDLE KNOWN MESSAGE CODES
	LD	A,(S2I_MSG)		; RETRIEVE THE MESSAGE BYTE
	BIT	7,A			; IDENTIFY ($80-$FF IS IDENTIFY MESSAGE)
	JR	NZ,S2I_MSG_IDENTIFY	; HANDLE IT
;
	; HANDLE UNKNOWN MESSAGE
	LD	A,$07			; MESSAGE REJECT
	LD	(S2I_MSG),A		; PUT IN MSG BUFFER
	JP	S2I_MSGIN		; RUN MESSAGE IN PHASE
;
S2I_MSGPROC4:
	POP	HL
	POP	DE
	POP	BC
	POP	AF
	LD	(S2I_MSG),A
	RET
;
; HANDLE IDENTIFY MESSAGE
;
S2I_MSG_IDENTIFY:
	AND	%00000111		; ISOLATE LUN BITS
	LD	(S2I_LUN),A		; SAVE ACTIVE LUN
	RET
;
;===============================================================
; SCSI COMMAND PROCESSOR
;===============================================================
;
; GET AND RESPOND TO COMMAND FROM INITIATOR
;
S2I_CMDPROC:
	CALL	S2I_MSGPROC		; CHECK/PROCESS HOST MESSAGES
	CALL	S2I_CMD		; GET THE COMMAND
	CALL	S2I_MSGPROC		; CHECK/PROCESS HOST MESSAGES
;
	; CHECK FOR UNIT INFO
	LD	A,(S2I_LUN)		; GET CURRENT LUN VALUE
	CP	$FF			; IS IT STILL NOT SET?
	JR	NZ,S2I_CMDPROC1		; IT IS SET, BYPASS
	LD	A,(S2I_CMDBUF + 1)	; GET SECOND BYTE OF CMD DATA
	RLCA				; SHIFT LUN BITS
	RLCA				; INTO
	RLCA				; POSTION
	AND	%00000111		; AND ISOLATE THEM
	LD	(S2I_LUN),A		; SAVE ACTIVE LUN
;
S2I_CMDPROC1:	; CHECK FOR VALID LUN
	LD	A,(S2I_CMDBUF + 0)	; GET THE OPCODE
	CP	$03			; REQUEST SENSE?
	JR	Z,S2I_CMDPROC2		; YES, ALLOW ANY LUN
	LD	A,(S2I_LUN)		; GET THE LUN VALUE BACK
	LD	B,A			; PUT IT IN B
	LD	A,S2I_LUNCNT - 1	; GET MAX LUN VALUE  IN A
	CP	B			; COMPARE ACT LUN (B) TO LUN COUNT (A)
	JR	NC,S2I_CMDPROC2		; MAX LUN VALUE NOT EXCEEDED, GOOD, CONTINUE
	LD	A,$02			; CHECK CONDITION STATUS VALUE
	LD	(S2I_STAT),A		; SAVE IT
	LD	A,$05			; ILLEGAL REQUEST SENSE BYTE
	LD	(S2I_SENSE),A		; SAVE IT
	JR	S2I_CMDPROC3		; TERMINATE THE COMMAND
;
S2I_CMDPROC2:	; PREP STACK FOR RETURN FROM HANDLER
	LD	HL,S2I_CMDPROC3
	PUSH	HL
;
	; DISPATCH BASED ON OPCODE
	LD	A,(S2I_CMDBUF + 0)	; GET THE OPCODE
	CP	$00			; TEST UNIT READY
	JP	Z,S2I_CMD_TESTREADY
	CP	$01			; REZERO UNIT
	JP	Z,S2I_CMD_REZERO
	CP	$03			; REQUEST SENSE
	JP	Z,S2I_CMD_SENSE
	CP	$04			; FORMAT UNIT
	JP	Z,S2I_CMD_FORMAT
	CP	$08			; GROUP 0 READ
	JP	Z,S2I_CMD_READ
	CP	$0A			; GROUP 0 WRITE
	JP	Z,S2I_CMD_WRITE
	CP	$12			; INQUIRY
	JP	Z,S2I_CMD_INQUIRY
	CP	$25			; READ CAPACITY
	JP	Z,S2I_CMD_CAPACITY
	CP	$28			; GROUP 1 READ
	JP	Z,S2I_CMD_G1RD
	CP	$2A			; GROUP 1 WRITE
	JP	Z,S2I_CMD_G1WR
	CP	$2F			; VERIFY
	JP	Z,S2I_CMD_VERIFY
;
	; UNKNOWN OPCODE
	LD	A,$02			; SET CHECK CONDITION VALUE
	LD	(S2I_STAT),A		; SAVE IT
	LD	A,$05			; ILLEGAL REQUEST SENSE BYTE
	LD	(S2I_SENSE),A		; SAVE IT
	RET
;
S2I_CMDPROC3:	; SEND COMMAND STATUS
;
	CALL	S2I_MSGPROC		; HANDLE ANY PENDING MESSAGES FROM INITIATOR

	CALL	S2I_STATUS		; SEND IT
	CALL	S2I_MSGPROC		; HANDLE ANY PENDING MESSAGES FROM INITIATOR
;
	; SEND COMMAND COMPLETE MESSAGE
	LD	A,$00			; COMMAND COMPLETE MESSAGE VALUE
	LD	(S2I_MSG),A		; PUT IN MSG BUFFER
	CALL	S2I_MSGIN		; RUN MESSAGE IN PHASE
	CALL	S2I_MSGPROC		; HANDLE ANY PENDING MESSAGES FROM INITIATOR
;
	RET				; DONE
;
; TEST UNIT READY COMMAND HANDLER
;
S2I_CMD_TESTREADY:
	; JUST RETURN WITH GOOD STATUS
	RET
;
; REZERO UNIT COMMAND HANDLER
;
S2I_CMD_REZERO:
	; JUST RETURN WITH GOOD STATUS
	RET
;
; REQUEST SENSE
;
S2I_CMD_SENSE:
;
	; SETUP THE RESPONSE BUFFER WITH SENSE RESPONSE TEMPLATE
	LD	HL,S2I_SENSERESP	; SOURCE
	LD	DE,S2I_RESPBUF		; DESTINATION
	LD	BC,S2I_SENSERESPLEN	; LENGTH
	LDIR				; COPY BYTES
;
	; FILL IN THE CURRENT SENSE BYTE
	LD	A,(S2I_SENSE)		; LOAD CURRENT SENSE BYTE
	LD	(S2I_RESPBUF + 2),A	; PUT IT IN THE RESPONSE
;
	; SEND THE RESPONSE
	LD	HL,S2I_RESPBUF
	LD	B,S2I_SENSERESPLEN
	LD	C,1
	CALL	S2I_DIN
	RET
;
; GROUP 0 READ DATA COMMAND HANDLER
;
S2I_CMD_READ:
	; SETUP HL:DE WITH LOGICAL BLOCK
	LD	H,0			; H IS NEVER USED, ALWAYS ZERO
	LD	A,(S2I_CMDBUF + 1)	; MSB
	AND	%00011111		; REMOVE IRRELEVANT LUN BITS
	LD	L,A			; LOAD IN L
	LD	A,(S2I_CMDBUF + 2)	; NEXT BYTE
	LD	D,A			; LOAD IN D
	LD	A,(S2I_CMDBUF + 3)	; LSB
	LD	E,A			; LOAD IN E
	; SETUP BC WITH TRANSFER COUNT (NUMBER OF BLOCKS)
	LD	A,(S2I_CMDBUF + 4)	; GET TRANSFER LENGTH
	LD	C,A			; COPY TO C
	LD	B,0			; COMPLETE BC, B := 0
;
	; READ THE DATA FROM IDE DEVICE
	LD	A,C			; RECOVER THE TRANSFER LENGTH
	OR	A			; SET FLAGS
	JP	NZ,S2I_READ		; NOT ZERO, SO DO READ AS IS
;
	; SPECIAL CASE IF TRANSFER LENGTH = 0, IT MEANS 256 BLOCKS!
	LD	B,1			; COUNT = 0, SO B := 1 FOR A 256 BLOCK XFR
	JP	S2I_READ		; NOW DO IT
;
; GROUP 0 WRITE DATA COMMAND HANDLER
;
S2I_CMD_WRITE:
	; SETUP HL:DE WITH LOGICAL BLOCK
	LD	H,0			; H IS NEVER USED, ALWAYS ZERO
	LD	A,(S2I_CMDBUF + 1)	; MSB
	AND	%00011111		; REMOVE IRRELEVANT LUN BITS
	LD	L,A			; LOAD IN L
	LD	A,(S2I_CMDBUF + 2)	; NEXT BYTE
	LD	D,A			; LOAD IN D
	LD	A,(S2I_CMDBUF + 3)	; LSB
	LD	E,A			; LOAD IN E
	; SETUP BC WITH TRANSFER COUNT (NUMBER OF BLOCKS)
	LD	A,(S2I_CMDBUF + 4)	; GET TRANSFER LENGTH
	LD	C,A			; COPY TO C
	LD	B,0			; COMPLETE BC, B := 0
;
	; WRITE DATA TO THE IDE DEVICE
	LD	A,C			; RECOVER THE TRANSFER LENGTH
	OR	A			; SET FLAGS
	JP	NZ,S2I_WRITE		; NOT ZERO, SO DO WRITE AS IS
;
	; SPECIAL CASE IF TRANSFER LENGTH = 0, IT MEANS 256 BLOCKS!
	LD	B,1			; COUNT = 0, SO B := 1 FOR A 256 BLOCK XFR
	JP	S2I_WRITE		; NOW DO IT
;
; INQUIRY COMMAND HANDLER
;
S2I_CMD_INQUIRY:
;
	; SETUP THE RESPONSE BUFFER WITH INQUIRY RESPONSE TEMPLATE
	LD	HL,S2I_INQRESP		; SOURCE
	LD	DE,S2I_RESPBUF		; DESTINATION
	LD	BC,S2I_INQRESPLEN	; LENGTH
	LDIR				; COPY BYTES
;
	; CHECK FOR VALID LUN, IF VALID SEND STD RESPONSE
	LD	A,(S2I_LUN)		; GET THE LUN VALUE BACK
	LD	B,A			; PUT IT IN B
	LD	A,S2I_LUNCNT - 1	; GET MAX LUN VALUE  IN A
	CP	B			; COMPARE ACT LUN (B) TO LUN COUNT (A)
	JR	NC,S2I_CMD_INQUIRY1	; MAX LUN VALUE NOT EXCEEDED, GOOD, CONTINUE
;
	; INVALID LUN, FIRST BYTE OF RESPONSE := $7F TO INDICATE SO
	LD	A,$7F			; A := $7F MEANS LUN NOT PRESENT
	LD	(S2I_RESPBUF),A		; SET FIRST BYTE OF RESPONSE
;
S2I_CMD_INQUIRY1:
;
	; FILL IN CONFIG INFO
	LD	A,(S2I_DEVICE)		; GET DEVICE
	LD	HL,STR_CFGIDE		; ASSUME IDE
	CP	DIODEV_PPIDE		; IS IT IDE?
	JR	Z,S2I_CMD_INQUIRY2	; YES, SET RESPONSE
	LD	HL,STR_CFGSD		; ASSUME SD
	CP	DIODEV_SD		; IS IT SD?
	JR	Z,S2I_CMD_INQUIRY2	; YES, SET RESPONSE
	LD	HL,STR_CFGUNK		; MUST BE UNKNOWN
S2I_CMD_INQUIRY2:
	LD	DE,S2I_RESPBUF + 24	; INJECT INTO PRODUCT ID
	LD	BC,3			; ALWAYS 3 BYTES
	LDIR				; UPDATE THE RESPONSE
;
	; FILL IN DEBUG FLAG IF ACTIVE
	LD	A,(S2I_DBGLVL)		; GET DEBUG LEVEL
	CP	2			; IS IT 2 OR GREATER?
	JR	C,S2I_CMD_INQUIRY3	; IF NOT, SKIP AHEAD
	LD	HL,STR_CFGDBG		; DEBUG STRING
	LD	DE,S2I_RESPBUF + 28	; INJECT INTO PRODUCT ID
	LD	BC,3			; ALWAYS 3 BYTES
	LDIR				; UPDATE THE RESPONSE
;
S2I_CMD_INQUIRY3:
;
	; SEND THE RESPONSE
	LD	HL,S2I_RESPBUF		; POINT TO RESPONSE BUFFER
	LD	A,(S2I_CMDBUF + 4)	; CMDBUF + 4 HAS ALLOC LEN
	LD	B,A			; MOVE IT TO B
	LD	C,1			; ALWAYS 1 OUTER LOOP ITERATION
	CALL	S2I_DIN			; SEND THE RESPONSE
	RET
;
; TEST UNIT READY COMMAND HANDLER
;
S2I_CMD_CAPACITY:
	LD	HL,S2I_CAPRESP512	; ASSUME 512 BYTE BLOCKS
	LD	A,(S2I_BLKSIZE)		; GET BLOCKSIZE
	CP	2			; 512?
	JR	Z,S2I_CMD_CAPACITY1	; DONE
	LD	HL,S2I_CAPRESP256	; ELSE 256 BYTE BLOCKS
S2I_CMD_CAPACITY1:
	LD	C,1			; 1 OUTER LOOP ITERATION
	LD	B,S2I_CAPRESPLEN
	CALL	S2I_DIN
	RET
;
; FORMAT UNIT COMMAND HANDLER
;
S2I_CMD_FORMAT:
	RET
;
; GROUP 1 READ DATA COMMAND HANDLER
;
S2I_CMD_G1RD:
	; SETUP HL:DE WITH LOGICAL BLOCK
	LD	A,(S2I_CMDBUF + 2)	; MSB
	LD	H,A			; LOAD IN H
	LD	A,(S2I_CMDBUF + 3)	; NEXT BYTE
	LD	L,A			; LOAD IN L
	LD	A,(S2I_CMDBUF + 4)	; NEXT BYTE
	LD	D,A			; LOAD IN D
	LD	A,(S2I_CMDBUF + 5)	; LSB
	LD	E,A			; LOAD IN E
	; SETUP BC WITH TRANSFER COUNT (NUMBER OF BLOCKS)
	LD	A,(S2I_CMDBUF + 7)	; GET TRANSFER LENGTH MSB
	LD	B,A			; COPY TO B
	LD	A,(S2I_CMDBUF + 8)	; GET TRANSFER LENGTH LSB
	LD	C,A			; COPY TO C
;
	; READ THE DATA FROM IDE DEVICE
	JP	S2I_READ
;
; GROUP 1 WRITE DATA COMMAND HANDLER
;
S2I_CMD_G1WR:
	; SETUP HL:DE WITH LOGICAL BLOCK
	LD	A,(S2I_CMDBUF + 2)	; MSB
	LD	H,A			; LOAD IN H
	LD	A,(S2I_CMDBUF + 3)	; NEXT BYTE
	LD	L,A			; LOAD IN L
	LD	A,(S2I_CMDBUF + 4)	; NEXT BYTE
	LD	D,A			; LOAD IN D
	LD	A,(S2I_CMDBUF + 5)	; LSB
	LD	E,A			; LOAD IN E
	; SETUP BC WITH TRANSFER COUNT (NUMBER OF BLOCKS)
	LD	A,(S2I_CMDBUF + 7)	; GET TRANSFER LENGTH MSB
	LD	B,A			; COPY TO B
	LD	A,(S2I_CMDBUF + 8)	; GET TRANSFER LENGTH LSB
	LD	C,A			; COPY TO C
;
	; READ THE DATA FROM IDE DEVICE
	JP	S2I_WRITE
;
; VERIFY COMMAND HANDLER
;
S2I_CMD_VERIFY:
	RET
;
;===============================================================
; READ STORAGE -> SCSI
;===============================================================
;
; READ AND SEND BLOCKS OF DATA FROM IDE TO SCSI
; BC = BLOCKS TO TRANSFER
; HL:DE = LOGICAL BLOCK TO START TRANSFERRING
;
S2I_READ:
;
	; CHECK FOR ZERO BLOCK XFR
	LD	A,B			; A : = B
	OR	C			; OR WITH C
	RET	Z			; ZERO BLOCKS, ALL DONE
;
	; OFFSET FOR LUN
	LD	A,(S2I_LUN)		; GET CURRENT LUN
	OR	A			; SET FLAGS
	JR	Z,S2I_READ2		; LUN 0 SELECTED, NO OFFSET NEEDED
	PUSH	BC			; SAVE BC SO WE CAN USE IT
	PUSH	DE			; SAVE DE SO WE CAN USE IT
	LD	B,A			; PUT CURRENT LUN IN B FOR COUNT
	LD	DE,S2I_LUN_OFFSET	; DE IS HIGH WORD OFFSET TO ADD PER LUN
S2I_READ1:	
	ADD	HL,DE			; DO IT
	DJNZ	S2I_READ1		; LOOP AS NEEDED
	POP	DE			; RECOVER DE
	POP	BC			; RECOVER BC
;
S2I_READ2:	; SAVE INCOMING PARMS
	PUSH	HL
	PUSH	DE
	PUSH	BC
;	
	; READ DATA FROM PPIDE DEVICE
	LD	B,BF_DIORD
	LD	A,(S2I_DEVICE)
	LD	C,A
	RST	08
	JR	NZ,S2I_READERR
;
	; SEND DATA TO SCSI BUS
	LD	HL,S2I_DSKBUF
	LD	B,0			; FULL 256 BYTES
	LD	A,(S2I_BLKSIZE)		; GET BLOCK SIZE
	LD	C,A			; AND USE FOR LOOP ITERATIONS
	CALL	S2I_DIN		; SEND DATA TO HOST
	CALL	S2I_MSGPROC		; HANDLE ANY PENDING MESSAGES FROM INITIATOR
;
	; RECOVER BC & HL:DE
	POP	BC
	POP	DE
	POP	HL
;
	; INCREMENT HL:DE
	INC	DE
	LD	A,D
	OR	E
	JR	NZ,S2I_READ3
	INC	HL
;
S2I_READ3:	; CHECK COUNT AND LOOP AS NEEDED
	DEC	BC
	LD	A,B
	OR	C
	JR	NZ,S2I_READ2
	RET
;
S2I_READERR:	; READ ERROR
	POP	AF			; UNWIND STACK
	POP	AF
	POP	AF
	LD	A,%00000010		; SET CHECK CONDITION STATUS
	LD	(S2I_STAT),A		; SAVE IT
	LD	A,4			; SENSE CODE 4 MEANS HARDWARE ERROR
	LD	(S2I_SENSE),A		; SAVE IT
	RET				; ABORT THE READ LOOP
;
;===============================================================
; WRITE SCSI -> STORAGE
;===============================================================
;
; READ SCSI DATA BLOCKS AND WRITE TO IDE DEVICE
; BC = BLOCKS TO TRANSFER
; HL:DE = LOGICAL BLOCK TO START TRANSFERRING
;
S2I_WRITE:
;
	; CHECK FOR ZERO BLOCK XFR
	LD	A,B			; A : = B
	OR	C			; OR WITH C
	RET	Z			; ZERO BLOCKS, ALL DONE
;
	; OFFSET FOR LUN
	LD	A,(S2I_LUN)		; GET CURRENT LUN
	OR	A			; SET FLAGS
	JR	Z,S2I_WRITE2		; LUN 0 SELECTED, NO OFFSET NEEDED
	PUSH	BC			; SAVE BC SO WE CAN USE IT
	PUSH	DE			; SAVE DE SO WE CAN USE IT
	LD	B,A			; PUT CURRENT LUN IN B FOR COUNT
	LD	DE,S2I_LUN_OFFSET	; DE IS HIGH WORD OFFSET TO ADD PER LUN
S2I_WRITE1:	
	ADD	HL,DE			; DO IT
	DJNZ	S2I_WRITE1		; LOOP AS NEEDED
	POP	DE			; RECOVER DE
	POP	BC			; RECOVER BC
;
S2I_WRITE2:	; SAVE INCOMING PARMS
	PUSH	HL
	PUSH	DE
	PUSH	BC
;
	; READ DATA FROM SCSI BUS
	LD	HL,S2I_DSKBUF
	LD	B,0			; FULL 256 BYTES
	LD	A,(S2I_BLKSIZE)		; GET BLOCK SIZE
	LD	C,A			; AND USE FOR LOOP ITERATIONS
	CALL	S2I_DOUT		; GET DATA FROM HOST
	CALL	S2I_MSGPROC		; HANDLE ANY PENDING MESSAGES FROM INITIATOR
;
	; RECOVER INCOMING PARMS
	POP	BC
	POP	DE
	POP	HL
;
	; RESAVE INCOMING PARMS
	PUSH	HL
	PUSH	DE
	PUSH	BC
;
	; WRITE DATA TO PPIDE DEVICE
	LD	B,BF_DIOWR
	LD	A,(S2I_DEVICE)
	LD	C,A
	RST	08			; PERFORM REQUESTED HBIOS CALL
	JR	NZ,S2I_WRITEERR
;
	; RECOVER INCOMING PARMS AGAIN
	POP	BC
	POP	DE
	POP	HL
;
	; INCREMENT HL:DE
	INC	DE
	LD	A,D
	OR	E
	JR	NZ,S2I_WRITE3
	INC	HL
;
S2I_WRITE3:	; CHECK COUNT AND LOOP AS NEEDED
	DEC	BC
	LD	A,B
	OR	C
	JR	NZ,S2I_WRITE2
	RET
;
S2I_WRITEERR:	; WRITE ERROR
	POP	AF			; UNWIND STACK
	POP	AF
	POP	AF
	LD	A,%00000010		; SET CHECK CONDITION STATUS
	LD	(S2I_STAT),A		; SAVE IT
	LD	A,4			; SENSE CODE 4 MEANS HARDWARE ERROR
	LD	(S2I_SENSE),A		; ABORT THE WRITE LOOP IMMEDIATELY
	RET
;
;===============================================================
; SCSI MESSAGE OUT PHASE
;===============================================================
;
; TRANSFER A MESSAGE BYTE FROM INITIATOR TO TARGET
; MESSAGE BYTE VALUE WILL BE STORED AT S2I_MSG
;
S2I_MSGOUT:
;
#IF (S2I_TRACE >= 2)
	LD	DE,STR_MSGOUT
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
#IFDEF S2I_IPDELAY
	CALL	S2I_IPDELAY
#ENDIF
;
	; TRANSFER THE BYTE
	LD	HL,S2I_MSG		; POINT TO MSG BYTE
	LD	BC,$0101		; SINGLE BYTE TRANSFER
	LD	A,S2I_PHMSGOUT		; MESSAGE OUT PHASE
	OUT	(S2I_TCR),A		; SET TARGET CONFIG
	JP	S2I_GET			; DO THE REAL WORK & RETURN
;
;===============================================================
; SCSI MESSAGE IN PHASE
;===============================================================
;
; TRANSFER A MESSAGE BYTE FROM TARGET TO INITIATOR
; MESSAGE BYTE VALUE MUST BE STORED AT S2I_MSG
;
S2I_MSGIN:
;
#IF (S2I_TRACE >= 2)
	LD	DE,STR_MSGIN
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
#IFDEF S2I_IPDELAY
	CALL	S2I_IPDELAY
#ENDIF
;
	; TRANSFER THE BYTE
	LD	HL,S2I_MSG		; POINT TO MSG BYTE
	LD	BC,$0101		; SINGLE BYTE TRANSFER
	LD	A,S2I_PHMSGIN		; MESSAGE IN PHASE
	OUT	(S2I_TCR),A		; SET TARGET CONFIG
	JP	S2I_PUT			; DO THE REAL WORK & RETURN
;
;===============================================================
; SCSI COMMAND PHASE
;===============================================================
;
; TRANSFER A COMMAND FROM INITIATOR TO TARGET
; RESULT IS STORED IN S2I_CMDBUF
;
S2I_CMD:
;
#IF (S2I_TRACE >= 2)
	LD	DE,STR_CMD
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
#IFDEF S2I_IPDELAY
	CALL	S2I_IPDELAY
#ENDIF
;
	; GET THE FIRST BYTE (OPCODE)
	LD	HL,S2I_CMDBUF		; POINT TO COMMAND BUFFER
	LD	A,$AA
	LD	(HL),A
	LD	BC,$0101		; JUST DO FIRST BYTE FOR NOW (OPCODE)
	LD	A,S2I_PHCMD		; COMMAND PHASE
	OUT	(S2I_TCR),A		; SET TARGET CONFIG
	CALL	S2I_GET			; DO IT
;
	; SET NUMBER OF BYTES LEFT TO GET BASED ON OPCODE
	LD	BC,$0001		; INITIALIZE COUNT (1 ITERATION OF ?)
	LD	A,(S2I_CMDBUF)		; GET OPCODE VALUE
	AND	%11100000		; ISLOATE OPCODE GROUP
	LD	B,5			; ASSUME GROUP 0, 6 BYTES (5 MORE)
	CP	%00000000		; GROUP 0?
	JR	Z,S2I_CMD1		; YES, DONE
	LD	B,9			; ASSUME GROUP 1, 10 BYTES (9 MORE)
	CP	%00100000		; GROUP 1?
	JR	Z,S2I_CMD1		; YES, DONE
	LD	B,11			; ASSUME GROUP 5, 12 BYTES (11 MORE)
	CP	%10100000		; GROUP 5?
	JR	Z,S2I_CMD1		; YES, DONE
	CALL	PANIC			; HUH???
;
S2I_CMD1:	; NOW GET THE REST OF THE BYTES
	LD	HL,S2I_CMDBUF + 1	; POINT TO COMMAND BUFFER + 1
	JP	S2I_GET			; DO IT & RETURN
;
;===============================================================
; SCSI STATUS PHASE
;===============================================================
;
; TRANSFER A STATUS BYTE FROM TARGET TO INITIATOR
; STATUS BYTE VALUE IS IN S2I_STATUS
;
S2I_STATUS:
;
#IF (S2I_TRACE >= 2)
	LD	DE,STR_STATUS
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
#IFDEF S2I_IPDELAY
	CALL	S2I_IPDELAY
#ENDIF
;
	; TRANSFER THE BYTE
	LD	HL,S2I_STAT		; POINT TO STATUS BYTE
	LD	BC,$0101		; SINGLE BYTE TRANSFER
	LD	A,S2I_PHSTATUS		; STATUS PHASE
	OUT	(S2I_TCR),A		; SET TARGET CONFIG
	JP	S2I_PUT			; DO THE REAL WORK & RETURN
;
;===============================================================
; SCSI DATA OUT PHASE
;===============================================================
;
; TRANSFER DATA FROM INITIATOR TO TARGET
; BC = BYTES TO RECEIVE
; HL = POINTER TO DATA BUFFER FOR RECEIVED DATA
;
S2I_DOUT:
;
#IF (S2I_TRACE >= 2)
	LD	DE,STR_DATAOUT
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
#IFDEF S2I_IPDELAY
	CALL	S2I_IPDELAY
#ENDIF
;
	; TRANSFER THE BYTES
	LD	A,S2I_PHDATAOUT		; DATA OUT PHASE
	OUT	(S2I_TCR),A		; SET TARGET CONFIG
	JP	S2I_GET			; DO THE REAL WORK & RETURN
;
;===============================================================
; SCSI DATA IN PHASE
;===============================================================
;
; TRANSFER DATA FROM TARGET TO INITIATOR
; BC = BYTES TO SEND
; HL = POINTER TO DATA BUFFER CONTAINING DATA TO SEND
;
S2I_DIN:
;
#IF (S2I_TRACE >= 2)
	LD	DE,STR_DATAIN
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,WRITESTR
#ENDIF
;
#IFDEF S2I_IPDELAY
	CALL	S2I_IPDELAY
#ENDIF
;
	; TRANSFER THE BYTES
	LD	A,S2I_PHDATAIN		; DATA IN PHASE
	OUT	(S2I_TCR),A		; SET TARGET CONFIG
	JP	S2I_PUT			; DO THE REAL WORK & RETURN
;
#IF (S2I_PDMA)
;
;===============================================================
; GET BYTES FROM SCSI CONTROLLER (TARGET MODE / PSEUDO DMA)
;===============================================================
;
; GET N BYTES FROM CONTROLLER -> MEMORY
; A = PHASE
; B = BYTES TO GET IN FIRST INNER LOOP
; C = NUMBER OF OUTER LOOP ITERATIONS
; HL = DESTINATION ADDRESS IN MEMORY
; AN APPROPRIATE PHASE MUST ALREADY BE SETUP ON SCSI CONTROLLER
;
S2I_GET:
;
	; SAVE PARMS FOR LATER
	PUSH	BC			; SAVE BC
	PUSH	HL			; SAVE HL
;
	; MAKE SURE THAT DATA BUS IS NOT ASSERTED
	IN	A,(S2I_ICR)		; GET ICR
	AND	%11111110		; TURNOFF ASSERT DATA BUS
	OUT	(S2I_ICR),A		; DO IT
;
	; TURN ON DMA PROCESSING & INITIATE TARGET MODE RECEIVE OPERATION
	IN	A,(S2I_MR)		; GET CURRENT MODE REGISTER VALUE
	OR	%00000010		; SET DMA MODE
	OUT	(S2I_MR),A		; DO IT
	OUT	(S2I_SDTR),A		; START DMA TARGET RECEIVE
;
	LD	E,C			; OUTER LOOP COUNTER TO E
	LD	C,S2I_IDR + $10		; PORT TO READ (BYTE IN W/ ACK)
;
	; LAST BYTE NEEDS SPECIAL HANDLING, PRE-DECRMENT AND DETECT
	; SINGLE BYTE TRANSFERS HERE
	DEC	B			; DECREMENT INNER LOOP COUNTER
	JR	NZ,S2I_GET1		; IF NOT ZERO, PROCEED TO NORMAL LOOP
	DEC	E			; B == 0, SO DECREMENT OUTER LOOP COUNTER
	JR	Z,S2I_GET2		; IF C == 0, ONE BYTE TRANSFER, SKIP AHEAD
;
S2I_GET1:	; WAIT FOR DMA REQ
	IN	A,(S2I_BSR)		; GET BSR
	BIT	6,A			; CHECK FOR DMA REQ
	JR	Z,S2I_GET1		; LOOP UNTIL DATA READY
;
	INI				; READ NEXT BYTE
;
	JR	NZ,S2I_GET1		; LOOP AS NEEDED
	DEC	E			; DECREMENT OUTER LOOP COUNTER
	JR	NZ,S2I_GET1		; LOOP AS NEEDED
;
S2I_GET2:	; WAIT FOR LAST BYTE
	IN	A,(S2I_BSR)		; GET BSR
	BIT	6,A			; CHECK FOR DMA REQ
	JR	Z,S2I_GET2		; LOOP UNTIL DATA READY
;
	; RESET MODE (CLEAR DMA MODE)
	IN	A,(S2I_MR)		; GET CURRENT MODE REGISTER VALUE
	AND	%11111101		; CLEAR DMA MODE
	OUT	(S2I_MR),A		; SEND IT
;
	; GET THE LAST BYTE
	INI
;
	; PRINT BUFFER RECEIVED
	POP	HL			; RECOVER HL
	POP	BC			; RECOVER BC
;
#IF (S2I_TRACE >= 2)
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,S2I_PRTBUF
#ENDIF
;
	RET				; DONE
;
;===============================================================
; PUT BYTES TO SCSI CONTROLLER (TARGET MODE / PSEUDO DMA)
;===============================================================
;
; PUT N BYTES FROM MEMORY -> CONTROLLER
; A = PHASE
; B = BYTES TO SEND IN FIRST INNER LOOP
; C = NUMBER OF OUTER LOOP ITERATIONS
; HL = POINTER TO DATA BUFFER CONTAINING DATA TO SEND
; AN APPROPRIATE PHASE MUST ALREADY BE SETUP ON SCSI CONTROLLER
;
S2I_PUT:
;
#IF (S2I_TRACE >= 2)
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,S2I_PRTBUF
#ENDIF
;
	; MAKE SURE DATA BUS IS BEING ASSERTED
	IN	A,(S2I_ICR)		; GET ICR
	OR	%00000001		; TURN ON ASSERT DATA BUS
	OUT	(S2I_ICR),A		; DO IT
;
	; TURN ON DMA PROCESSING & INITIATE TARGET MODE SEND OPERATION
	IN	A,(S2I_MR)		; GET CURRENT MODE REGISTER VALUE
	OR	%00000010		; SET DMA MODE
	OUT	(S2I_MR),A		; DO IT
	OUT	(S2I_SDS),A
;
	LD	E,C			; OUTER LOOP COUNTER TO E
	LD	C,S2I_IDR + $10		; PORT TO WRITE (BYTE OUT W/ ACK)
;
S2I_PUT1:	; WAIT FOR DMA REQ
	IN	A,(S2I_BSR)		; GET BSR
	BIT	6,A			; CHECK FOR DMA REQ
	JR	Z,S2I_PUT1		; LOOP UNTIL DATA READY
;
	OUTI				; SEND NEXT BYTE
;
	JR	NZ,S2I_PUT1		; LOOP AS NEEDED
	DEC	E			; DECREMENT OUTER LOOP COUNTER
	JR	NZ,S2I_PUT1		; LOOP AS NEEDED
;
S2I_PUT2:	; WAIT FOR LAST BYTE TO BE SENT
	IN	A,(S2I_BSR)		; GET BSR
	BIT	6,A			; CHECK FOR DMA REQ
	JR	Z,S2I_PUT2		; LOOP UNTIL DATA READY
;
	; RESET MODE (CLEAR DMA MODE)
	IN	A,(S2I_MR)		; GET CURRENT MODE REGISTER VALUE
	AND	%11111101		; CLEAR DMA MODE
	OUT	(S2I_MR),A		; SEND IT
;
	; DEASSERT DATA BUS
	IN	A,(S2I_ICR)		; GET ICR
	AND	%11111110		; TURNOFF ASSERT DATA BUS
	OUT	(S2I_ICR),A		; DO IT
;
#IF (S2I_TRACE >= 2)
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,PC_PERIOD
#ENDIF
;
	RET
#ELSE
;
;===============================================================
; GET BYTES FROM SCSI CONTROLLER (TARGET MODE / POLLING)
;===============================================================
;
; GET N BYTES FROM CONTROLLER -> MEMORY
; A = PHASE
; B = BYTES TO GET IN FIRST INNER LOOP
; C = NUMBER OF OUTER LOOP ITERATIONS
; HL = DESTINATION ADDRESS IN MEMORY
; AN APPROPRIATE PHASE MUST ALREADY BE SETUP ON SCSI CONTROLLER
;
S2I_GET:
;
	; SAVE PARMS FOR LATER
	PUSH	BC			; SAVE BC
	PUSH	HL			; SAVE HL
;
	; MAKE SURE THAT DATA BUS IS NOT ASSERTED
	IN	A,(S2I_ICR)		; GET ICR
	AND	%11111110		; TURNOFF ASSERT DATA BUS
	OUT	(S2I_ICR),A		; DO IT
;
	LD	E,C			; OUTER LOOP COUNTER TO E
	LD	C,S2I_CDR		; PORT TO READ
;
S2I_GET1:	; ASSERT REQ TO REQUEST A BYTE
	IN	A,(S2I_TCR)		; GET TCR
	OR	%00001000		; SET REQ
	OUT	(S2I_TCR),A		; DO IT
;
S2I_GET2:	; WAIT FOR ACK
	IN	A,(S2I_BSR)		; GET BUS AND STATUS
	AND	%00000001		; ISOLATE ACK
	JR	Z,S2I_GET2		; LOOP TILL ACK ASSERTED
;
	; READ AND STORE BYTE
	INI				; READ AND INCREMENT
	PUSH	AF			; SAVE STATE
;
	; DEASSERT REQ
	IN	A,(S2I_TCR)		; GET TCR
	AND	%11110111		; CLEAR REQ
	OUT	(S2I_TCR),A		; DO IT
;
S2I_GET3:	; WAIT FOR ACK TO BE DEASSERTED
	IN	A,(S2I_BSR)		; GET BUS AND STATUS
	AND	%00000001		; ISOLATE ACK
	JR	NZ,S2I_GET3		; LOOP TILL ACK DEASSERTED
;
	POP	AF			; RESTORE STATE
	JR	NZ,S2I_GET1		; INNER LOOP
	DEC	E			; DECREMENT OUTER LOOP COUNTER
	JR	NZ,S2I_GET1		; OUTER LOOP
;
	; PRINT BUFFER RECEIVED
	POP	HL			; RECOVER HL
	POP	BC			; RECOVER BC
;
#IF (S2I_TRACE >= 2)
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,S2I_PRTBUF
#ENDIF
;
	RET				; DONE
;
;===============================================================
; PUT BYTES TO SCSI CONTROLLER (TARGET MODE / POLLING)
;===============================================================
;
; PUT N BYTES FROM MEMORY -> CONTROLLER
; A = PHASE
; B = BYTES TO SEND IN FIRST INNER LOOP
; C = NUMBER OF OUTER LOOP ITERATIONS
; HL = POINTER TO DATA BUFFER CONTAINING DATA TO SEND
; AN APPROPRIATE PHASE MUST ALREADY BE SETUP ON SCSI CONTROLLER
;
S2I_PUT:
;
#IF (S2I_TRACE >= 2)
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,S2I_PRTBUF
#ENDIF
;
	; START ASSERTING DATA BUS
	IN	A,(S2I_ICR)		; GET ICR
	OR	%00000001		; TURN ON ASSERT DATA BUS
	OUT	(S2I_ICR),A		; DO IT
;
	LD	E,C			; OUTER LOOP COUNTER TO E
	LD	C,S2I_CDR		; PORT TO WRITE
;
S2I_PUT1:	; ASSERT DATA ON BUS AND ASSERT REQ
	OUTI				; SET OUTPUT VALUE OF NEXT BYTE
	PUSH	AF			; SAVE STATE
;
	IN	A,(S2I_TCR)		; GET TCR
	OR	%00001000		; SET REQ
	OUT	(S2I_TCR),A		; DO IT
;
S2I_PUT2:	; WAIT FOR ACK, THEN DEASSERT REQ
	IN	A,(S2I_BSR)		; GET BUS AND STATUS
	AND	%00000001		; ISOLATE ACK
	JR	Z,S2I_PUT2		; LOOP TILL ACK ASSERTED
	IN	A,(S2I_TCR)		; GET TCR
	AND	%11110111		; CLEAR REQ
	OUT	(S2I_TCR),A		; DO IT
;
S2I_PUT3:	; WAIT FOR ACK TO BE DEASSERTED
	IN	A,(S2I_BSR)		; GET BUS AND STATUS
	AND	%00000001		; ISOLATE ACK
	JR	NZ,S2I_PUT3		; LOOP TILL ACK DEASSERTED
;
	; LOOP FOR NEXT BYTE
	POP	AF			; RESTORE STATE
	JR	NZ,S2I_PUT1		; LOOP AS NEEDED
	DEC	E			; DECREMENT OUTER LOOP COUNTER
	JR	NZ,S2I_PUT1		; LOOP AS NEEDED
;
	; DEASSERT DATA BUS
	LD	A,%00001000		; DEASSERT DATA LINES
	OUT	(S2I_ICR),A		; SEND IT
;
#IF (S2I_TRACE >= 2)
	LD	A,(S2I_DBGLVL)
	CP	2
	CALL	NC,PC_PERIOD
#ENDIF
;
	RET
;
#ENDIF
;
;===============================================================
; ZERO THE FIRST 256 BLOCKS OF IDE STORAGE
;===============================================================
;
S2I_ZERO:
	LD	DE,STR_ZERO		; ANNOUNCEMENT
	CALL	WRITESTR		; PRINT IT
;
	; INIT THE HBIOS DISK BUFFER WITH ALL ZEROES
	LD	HL,S2I_ZEROSEC		; SOURCE OF SECTOR DATA (ALL ZEROES)
	LD	DE,S2I_DSKBUF		; DESTINATION IS DISK BUFFER
	LD	BC,512			; LENGTH OF A SECTOR
	LDIR				; COPY IT
;
	; SETUP LOOP
	LD	B,0			; B IS LOOP COUNTER
	LD	HL,0			; START WITH BLOCK 0
	LD	DE,0			; START WITH BLOCK 0
;
S2I_ZERO1:	; LOOP, SAVE WORKING VARIABLES
	PUSH	HL			; SAVE HL
	PUSH	DE			; SAVE DE
	PUSH	BC			; SAVE BC
;
	; DO THE HBIOS WRITE
	LD	B,BF_DIOWR		; HBIOS WRITE FUNCTION
	LD	A,(S2I_DEVICE)
	LD	C,A
	RST	08			; DO IT
;
	CALL	PC_PERIOD		; PROGRESS FEEDBACK
;
	; RECOVER WORKING VARIABLES
	POP	BC			; RECOVER BC
	POP	DE			; RECOVER DE
	POP	HL			; RECOVER HL
;
	; INCREMENT BLOCK AND LOOP AS NEEDED
	INC	DE			; NEXT BLOCK
	DJNZ	S2I_ZERO1		; LOOP UNTIL DONE
;
	; DONE
	RET
;
;===============================================================
; READ A BLOCK OF DATA INTO BUFFER
;===============================================================
;
S2I_RDBLK:
	LD	DE,STR_RDBLK		; ANNOUNCEMENT
	CALL	WRITESTR		; PRINT IT
;
	; DO THE HBIOS READ
	LD	B,BF_DIORD		; HBIOS READ FUNCTION
	LD	A,(S2I_DEVICE)
	LD	C,A
	LD	HL,0			; BLOCK 0
	LD	DE,0			; BLOCK 0
	RST	08			; DO IT
;
	RET
;
;===============================================================
; WRITE A BLOCK OF DATA INTO BUFFER
;===============================================================
;
S2I_WRBLK:
;
	LD	DE,STR_WRBLK		; ANNOUNCEMENT
	CALL	WRITESTR		; PRINT IT
;
	; DO THE HBIOS WRITE
	LD	HL,0			; BLOCK 0
	LD	DE,0			; BLOCK 0
	LD	B,BF_DIOWR		; HBIOS WRITE FUNCTION
	LD	A,(S2I_DEVICE)
	LD	C,A
	RST	08			; DO IT
;
	RET
;
;===============================================================
; DISPLAY BUFFER DATA
;===============================================================
;
	LD	DE,STR_DUMP		; ANNOUNCEMENT
	CALL	WRITESTR		; PRINT IT
;
S2I_DUMP:
	LD	HL,S2I_DSKBUF		; SET ADDRESS TO DUMP
	LD	DE,S2I_DSKBUF + $200	; SET END ADDRESS

DUMP_BUFFER:
	CALL	NEWLINE			; NEXT LINE
BLKRD:
	CALL	PHL			; PRINT START LOCATION
	LD	C,16			; SET FOR 16 LOCS
	PUSH	HL			; SAVE STARTING HL
NXTONE:
	LD 	A,(HL)			; GET BYTE
	CALL	PRTHEXBYTE		; PRINT IT
	CALL	PC_SPACE		;
UPDH:	
	INC	HL			; POINT NEXT
	DEC	C			; DEC. LOC COUNT
	JR	NZ,NXTONE		; IF LINE NOT DONE
					; NOW PRINT 'DECODED' DATA TO RIGHT OF DUMP
PCRLF:
	CALL	PC_SPACE		; SPACE IT
	LD	C,16			; SET FOR 16 CHARS
	POP	HL			; GET BACK START
PCRLF0:
	LD	A,(HL)			; GET BYTE
	AND	060H			; SEE IF A 'DOT'
	LD	A,(HL)			; O.K. TO GET
	JR	NZ,PDOT			;
DOT:
	LD	A,2EH			; LOAD A DOT	
PDOT:
	CALL	COUT			; PRINT IT
	INC	HL			; 
	LD	A,D			;
	CP	H			;
	JR	NZ,UPDH1		;
	LD	A,E			;
	CP	L			;
	JP	Z,DUMP_END		;
;
;IF BLOCK NOT DUMPED, DO NEXT CHARACTER OR LINE
UPDH1:
	DEC	C			; DEC. CHAR COUNT
	JR	NZ,PCRLF0		; DO NEXT
CONTD:
	CALL	NEWLINE			; NEXT LINE
	JP	BLKRD			;

DUMP_END:
	RET
;
PHL:
	LD	A,H			; GET HI BYTE
	CALL	PRTHEXBYTE		; DO HEX OUT ROUTINE
	LD	A,L			; GET LOW BYTE
	CALL	PRTHEXBYTE		; HEX IT
	CALL	PC_SPACE		; 
	RET				; DONE  
;
;===============================================================
; HELPER ROUTINES
;===============================================================
;
; KEYCHK CHECKS FOR A KEYPRESS, RETURNS KEY VALUE IF PRESSED, OTHERWISE ZERO
;
S2I_KEYCHK:
	CALL	CST	; KEY PENDING?
	OR	A	; SET FLAGS
	RET	Z	; RETURN IF NO KEY WAITING
	CALL	CIN	; GET THE KEY
	OR	A	; SET FLAGS
	RET		; DONE
;
; PRINT A SNAPSHOT OF KEY SCSI CONTROLLER REGISTER VALUES
; [CBS:BSR:CDR:MR]
;
S2I_SNAP:
;
	; SAVE CPU REGISTERS
	PUSH	AF		; SAVE AF
	PUSH	BC		; SAVE BC
	PUSH	DE		; SAVE DE
	PUSH	HL		; SAVE HL
;
	; READ KEY REGISTERS AS QUICKLY AS POSSIBLE
	IN	A,(S2I_CBS)	; GET CBS
	LD	B,A		; SAVE IN B
	IN	A,(S2I_BSR)	; GET BSR
	LD	C,A		; SAVE IN C
	IN	A,(S2I_CDR)	; GET CDR
	LD	D,A		; SAVE IN D
	IN	A,(S2I_MR)	; GET MR
	LD	E,A		; SAVE IN E
;
	; PRINT THE REGISTERS
	CALL	PC_SPACE	; SPACER
	CALL	PC_LBKT		; MARKER
	LD	A,B		; RECOVER CBS
	CALL	PRTHEXBYTE	; PRINT CBS
	CALL	PC_COLON	; SEPARATOR
	LD	A,C		; RECOVER BSR
	CALL	PRTHEXBYTE	; PRINT BSR
	CALL	PC_COLON	; SEPARATOR
	LD	A,D		; RECOVER CDR
	CALL	PRTHEXBYTE	; PRINT CDR
	CALL	PC_COLON	; SEPARATOR
	LD	A,E		; RECOVER MR
	CALL	PRTHEXBYTE	; PRINT MR
	CALL	PC_RBKT		; MARKER
;
	; RESTORE CPU REGISTERS AND RETURN
	POP	HL		; RECOVER HL
	POP	DE		; RECOVER DE
	POP	BC		; RECOVER BC
	POP	AF		; RECOVER AF
;
	RET
;
; PRINT A BUFFER AT HL THAT IS BC BYTES LONG
; PRINT NO MORE THAN 16 BYTES
;
S2I_PRTBUF:
;
	; SAVE STATE
	PUSH	AF
	PUSH	BC
	PUSH	DE
	PUSH	HL
;
	; SETUP E AS LIMIT COUNTER
	LD	E,$10		; 16 BYTES MAX
;
S2I_PRTBUF1:
	; PRINT NEXT BYTE AND INCREMENT POINTER
	LD	A,(HL)
	CALL	PRTHEXBYTE
	CALL	PC_SPACE
	INC	HL
;
	; MANAGE COUNTERS AND LOOP AS NEEDED
	DEC	E		; DECREMENT LIMIT COUNTER
	JR	Z,S2I_PRTBUF2	; DONE IF ZERO
	DEC	B		; DECREMENT INNER LOOP COUNTER
	JR	NZ,S2I_PRTBUF1	; DO INNER LOOP
	DEC	C		; DECREMENT OUTER LOOP COUNTER
	JR	NZ,S2I_PRTBUF1	; DO OUTER LOOP
;
S2I_PRTBUF2:
;
	; RESTORE STATE
	POP	HL
	POP	DE
	POP	BC
	POP	AF
;
	RET
;
;
;
S2I_DELAY512:
	CALL	$+3
S2I_DELAY256:
	CALL	$+3
S2I_DELAY128:
	CALL	$+3
S2I_DELAY64:
	CALL	$+3
S2I_DELAY32:
	CALL	$+3
S2I_DELAY16:
	CALL	$+3
S2I_DELAY8:
	CALL	$+3
S2I_DELAY4:
	CALL	$+3
S2I_DELAY2:
	CALL	$+3
S2I_DELAY1:
	RET
;
; UTILITY ROUTINES
;
#DEFINE	CIOMODE_HBIOS
#INCLUDE "util.asm"
;
;===============================================================
; STATIC STORAGE
;===============================================================
;
STR_BANNER	.DB	"\r\n\r\n=== S2I Target Command Processor V", S2I_VERTAG, " ==="
		.DB	"\r\nPress <esc> for monitor$"
STR_DONE	.DB	"\r\n\r\nTerminating...$"
STR_SELECT	.DB	"\r\n\r\nWaiting for selection...$"
STR_CMD		.DB	"\r\nCommand: $"
STR_DATAIN	.DB	"\r\nData In: $"
STR_DATAOUT	.DB	"\r\nData Out: $"
STR_STATUS	.DB	"\r\nStatus: $"
STR_MSGIN	.DB	"\r\nMessage In: $"
STR_MSGOUT	.DB	"\r\nMessage Out: $"
STR_BUSFREE	.DB	"\r\n\r\n*** Bus Free ***$"
STR_ZERO	.DB	"\r\nZero storage...\r\n$"
STR_RDBLK	.DB	"\r\nRead block from storage...\r\n$"
STR_WRBLK	.DB	"\r\nWrite block to storage...\r\n$"
STR_DUMP	.DB	"\r\nDump data buffer...\r\n$"
STR_CONFIG	.DB	"\r\nConfig Byte=0x$"
STR_CONFIG1	.DB	"\r\nSCSI ID=$"
STR_CONFIG2	.DB	", Block Size=$"
STR_CONFIG3	.DB	"\r\nLUN COUNT=$"
STR_CONFIG4	.DB	", LUN SIZE=$"
STR_CONFIG5	.DB	"MB$"
STR_BLK512	.DB	"512 bytes$"
STR_BLK256	.DB	"256 bytes$"
STR_BLKUNK	.DB	"Unknown$"
STR_DBGLVL	.DB	"\r\nDebug Level: $"
;
#IF (S2I_PDMA)
STR_XFRMODE	.DB	"\r\nUsing Pseudo DMA I/O$"
#ELSE
STR_XFRMODE	.DB	"\r\nUsing Polled I/O$"
#ENDIF
;
STR_DEVIDE	.DB	"\r\nData storage on IDE device$"
STR_DEVSD	.DB	"\r\nData storage on SD Card device$"
STR_DEVUNK	.DB	"\r\nData storage on Unknown device$"
;
STR_CFGIDE	.DB	"IDE"
STR_CFGSD	.DB	"SD "
STR_CFGUNK	.DB	"???"
STR_CFGDBG	.DB	"DBG"
;
;===============================================================
; DEVICE PERSONALITIES (CUSTOMIZED RESPONSE DATA)
;===============================================================
;
; N8VEM PERSONALITY (GENERIC 20MB DIRECT ACCESS STORAGE DEVICE)
;
#IF (S2I_PER == S2I_PER_N8VEM)
;
S2I_INQRESP		.DB	$00,$00,$01,$01
		 	.DB	032,$00,$00,$00		; ADDITIONAL LENGTH
			.DB	"N8VEM   "		; VENDOR ID
			.DB	"S2I     "		; PRODUCT ID
			.DB	"        "		; PRODUCT ID (CONT)
			.DB	S2I_VERTAG		; PRODUCT REVISION
S2I_INQRESPLEN		.EQU	$-S2I_INQRESP
;
S2I_MAXBLK		.EQU	(S2I_LUNSIZ * 1024 * 2) - 1
S2I_CAPRESP512:
			.DB	(S2I_MAXBLK >> 24) & $FF
			.DB	(S2I_MAXBLK >> 16) & $FF
			.DB	(S2I_MAXBLK >> 8) & $FF
			.DB	S2I_MAXBLK & $FF
			.DB	$00,$00,$02,$00		; BLOCK SIZE IN BYTES
S2I_CAPRESPLEN		.EQU	$-S2I_CAPRESP512
S2I_CAPRESP256:
			.DB	(S2I_MAXBLK >> 24) & $FF
			.DB	(S2I_MAXBLK >> 16) & $FF
			.DB	(S2I_MAXBLK >> 8) & $FF
			.DB	S2I_MAXBLK & $FF
			.DB	$00,$00,$01,$00		; BLOCK SIZE IN BYTES
;
#ENDIF
;
; SEAGATE ST125N PERSONALITY
;
#IF (S2I_PER == S2I_PER_ST125N)
;
S2I_INQRESP		.DB	$00, $00, $01, $01, $3C, $00, $00, $00, $53, $45, $41, $47, $41, $54, $45, $20	
			.DB	$53, $54, $31, $32, $35, $4E, $20, $20, $20, $20, $20, $20, $20, $20, $20, $20	
			.DB	$04, $78, $6A, $00, $20, $20, $31, $31, $31, $32, $34, $33, $31, $00, $00, $00	
			.DB	$00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $6C, $6C, $6C, $6C, $6C, $6C	
			.DB	$6C, $6C, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00, $00	
			.DB	$00, $00, $00, $00, $00, $00, $00, $00, $00
S2I_INQRESPLEN		.EQU	$-S2I_INQRESP
;
S2I_CAPRESP512		.DB	$00,$00,$A3,$C0		; LOGICAL BLOCK COUNT
			.DB	$00,$00,$02,$00		; BLOCK SIZE IN BYTES
S2I_CAPRESPLEN		.EQU	$-S2I_CAPRESP512
S2I_CAPRESP256		.DB	$00,$00,$A3,$C0		; LOGICAL BLOCK COUNT
			.DB	$00,$00,$01,$00		; BLOCK SIZE IN BYTES
;
#ENDIF
;
S2I_SENSERESP		.DB	$70,$00,$00,$00
			.DB	$00,$00,$00,$00
S2I_SENSERESPLEN	.EQU	$-S2I_SENSERESP
;
S2I_ZEROSEC		.FILL	512,0
;
	.FILL	$6000 - $
;
;===============================================================
; WORKING DATA (DYNAMIC STORAGE)
;===============================================================
;
; WORKING DATA STARTS AT $9000
;
	.ORG	$9000
;
S2I_MSG		.DS	1	; MESSAGE BYTE BUFFER
S2I_SENSE	.DS	1	; SENSE BYTE BUFFER
S2I_STAT	.DS	1	; CURRENT STATUS
S2I_LUN		.DS	1	; ACTIVE LUN
S2I_SID		.DS	1	; SCSI BUS DEVICE ID
S2I_SIDMASK	.DS	1	; SCSI DEVICE ID BIT MASK
S2I_CONFIG	.DS	1	; SCSI CONFIG BYTE
S2I_BLKSIZE	.DS	1	; BLOCK SIZE (1=256, 2=512)
S2I_DEVICE	.DS	1	; STORAGE DEVICE (HBIOS DEVICE ID)
S2I_DBGLVL	.DS	1	; DEBUG LEVEL (0=NONE, 1=ERROR, 2=INFO)
;
S2I_CMDLEN	.DS	1	; LENGTH OF COMMAND DATA RECIEVED
S2I_CMDBUF	.DS	12	; UP TO 12 BYTES FOR RECEIVED COMMAND DATA
;
S2I_RESPLEN	.DS	1	; LENGTH OF RESPONSE DATA
S2I_RESPBUF	.DS	256	; UP TO 256 BYTES FOR RESPONSE
;
S2I_CBS_PRV	.DS	1
S2I_BSR_PRV	.DS	1
;
S2I_DSKBUF	.DS	512	; PHYSICAL DISK BUFFER
;
;===============================================================
;
	.END
