;	8080 CBIOS for CP/Mega88 -- brainchild of Takashi Toyoshima
;
;	Author Rienk Koolstra
;
;  altough Z80 Mnemonics are used, the code is strictly i8080.
; 
; Version 1.2 

ccp		equ 0E400h	; Base of CCP
bdos		equ ccp + 0806h	; Base of BDOS.
bios		equ ccp + 1600h	; Base of BIOS.


; CP/M base page data, vector and buffer Addresses.
iobyte		equ 03h		; Intel standard I/O definition byte.
userdrv		equ 04h		; Current user number and drive.
dmabuf		equ 80h		; Default disk buffer

;	I/O ports
CON_ST	equ	0		; console status port
CON_D	equ	1		; console data port
LST_ST	equ	2		; printer status port
LST_D	equ	3		; printer data port
AUX_D	equ	5		; auxiliary data port
SD_DSK	equ	10		; disk drive
SD_TKL	equ	11		; disk track low
SD_TKH	equ	12		; disk track high
SD_SEC	equ	13		; disk sector
SD_CMD	equ	14		; disk command
SD_STAT	equ	15		; disk status
DMA_L	equ	16		; dma address low
DMA_H	equ	17		; dma address high
BANK	equ	18		; select memory bank

; Text formatters
LF	equ	0AH		; line feed
CR	equ	0DH		; carriage return
CLS	equ	0Ch		; clear screen (FF)


; ============================================================

		ORG	bios		; BIOS origin.

; ============================================================
; BIOS jump table.
; ============================================================
	jp boot		; 0 Initialize.
wboote:	jp wboot	; 1 Warm boot.
	jp const	; 2 Console status.
	jp conin	; 3 Console input.
	jp conout	; 4 Console output.
	jp list		; 5 List output.
	jp punch	; 6 punch output.
	jp reader	; 7 Reader input.
	jp home		; 8 Home disk.
	jp seldsk	; 9 Select disk.
	jp settrk	; 10 Select track.
	jp setsec	; 11 Select sector.
	jp setdma	; 12 Set DMA Address.
	jp read		; 13 Read 128 bytes.
	jp write	; 14 Write 128 bytes.
	jp listst	; 15 List status.
	jp sectran	; 16 Sector translate.

; ============================================================
; Disk parameter headers for disk A - C
; ============================================================
dpbase:	dw 0000h,0000h,0000h,0000h,dirbuf,dpbF8,0000h,alv00
dpB:	dw 0000h,0000h,0000h,0000h,dirbuf,dpbF8,0000h,alv01
dpC:	dw 0000h,0000h,0000h,0000h,dirbuf,dpbF8,0000h,alv02

dpbF8:	; 8 MB flash disk 
	dw 128		; SPT - sectors per track
	db 5		; BSH - block shift factor
	db 31		; BLM - block mask
	db 1		; EXM - Extent mask
	dw 2043		; DSM - Storage size (blocks - 1)
	dw 511		; DRM - Number of directory entries - 1
	db 11110000B	; AL0 - 1 bit set per directory block
	db 00000000B	; AL1 -            "
	dw 0		; CKS - DIR check vector size (DRM+1)/4 (0=fixed disk)
	dw 1		; OFF - Reserved tracks	


; ============================================================
; Warm boot
; ============================================================

wboot:
	ld SP,dmabuf		; use space below DMA buffer for stack

;	Load  fresh copy of the CCP

	ld C,0			; select disk 0
	call seldsk
	call home		; go to track 00
	ld B,16			; ccp is 16 sectors (2k) long
	inc  C			; start sector (0 = boot sector)
	ld HL,ccp		; destination
	ld DE,128		; blocksize

load:	push BC			; save sector counters
	call setsec		; set sector from register C
	push HL			; dma address ...
	pop BC			; ... to BC
	call setdma		; set dma address
	call read		; transfer sector to dma
	add HL,DE		; point to next memory block
	pop BC			; get sector counters
	dec B			; sectors to go
	jp z,gocpm		; if done, go play
	inc C			; if not, get next sector
	jp load			; repeat...

; ============================================================
; Common code for cold and warm boot
; ============================================================

gocpm:
	ld A,0C3h		; opcode for 'jp'
	ld (0),A		; load at start of ram.
	ld HL,wboote		; address of jump for a warm boot.
	ld (0001h),HL		; 0000 jp wboote
;                                
	ld (0005h),A		; opcode for 'jp'.
	ld HL,bdos		; address of jump to bdos.
	ld (0006h),HL		; 0005 jp bdos
;                                
	ld BC,dmabuf		; address of bios dma buffer.
	call setdma              
;                                
	ld A,(userdrv)		; get current drive number
	ld C,A			; pass drive number in c.
	jp ccp			; start cp/m by jumping to the ccp.
                                 
; ============================================================
; Console I/O routines
; ============================================================


const:
	in A,(CON_ST)		; get console status
	ret

conin:
	in A,(CON_D)		; get character from console
	ret

conout:
	ld A,C			; character to accumulator
	out (CON_D),A		; send character to console
	ret

;  list, punch & reader lead to dummy devices. outputs are sinks

list:
	ld A,C			; character to accumulator
	out (LST_D),A		; send character to printer
	ret

listst:
	in A,(LST_ST)		; get printer status
	ret

punch:
	ld A,C			; character to accumulator
	out (AUX_D),A		; send character to punch
	ret

reader:
	in A,(AUX_D)		; get character from reader
	ret

; ============================================================
; Disk processing entry points
; ============================================================

seldsk: ld HL,0000h		; error return code
	ld A,C			; target from C
	cp 3			; disk A-C?
	ret nc			; no, error
	ld B,0
	ld HL,mnttab
	add HL,BC
	ld A,(HL)
	out (SD_DSK),A		; select disk volume
	ld L,C			; l=disk number 0,1,2,3
	ld H,0
	add HL,HL		; *2
	add HL,HL		; *4
	add HL,HL		; *8
	add HL,HL		; *16 (size of each header)
	ld DE,dpbase
	add HL,DE		; hl=dpbase+(diskno*16)
	ret


; ------------------------------------------------------------
home:
	ld BC,0			; select track 0
				; continue with settrk

; ------------------------------------------------------------
settrk: 
	ld A,C			; track from bdos in register BC.
	out (SD_TKL),A		; send low byte
	ld A,B
	out (SD_TKH),A		; send high byte
	ret

; ------------------------------------------------------------
setsec:
	ld A,C
	out (SD_SEC),A
	ret

; ------------------------------------------------------------
sectran:			; no translation:
	push BC			; copy BC ...
	pop HL			; ... to HL
	ret

; ------------------------------------------------------------
setdma:
	ld A,C			; low order address
	out (DMA_L),A
	ld A,B			; high order address
	out (DMA_H),A		; in dma
	ret

; ------------------------------------------------------------
read:
	xor A			; read command (0) -> a
	jp diskio		; perform the actual i/o

; ------------------------------------------------------------
write:
	ld A,1			; write command (1) -> a


diskio:
	out (SD_CMD),A		; start i/o operation
	in A,(SD_STAT)		; get status
	ret

; ------------------------------------------------------------
; Data storage

mnttab:		; disk volume mount table
	db -1	; A:
	db -1	; B:
	db -1	; C:

dirbuf:	ds 128			; scratch directory area
alv00:	equ $ 			; allocation vector 0

; this data will be overwritten by disk A: allocation vectors

SIGNON: db CR,CLS,'CP/Mega88 vm by Takashi Toyoshima',CR,LF
	db '59K CP/M 2.2 banked BIOS V1.2 (i8080)',CR,LF
SGNCP:	db 'CP/M 2.2 (c) 1979 by Digital Research',CR,LF
	db LF,0

boot:	di
	ld A,(100h)
	ld (mnttab),A
	inc A
	ld (mnttab+1),A
	inc A
	ld (mnttab+2),A
	xor A			; make sure we ase using bank 0
	out (BANK),A		; before setting stackpointer

	ld SP,dmabuf		; use space below DMA buffer for stack
	ld HL,SIGNON		; print message
	call print
	xor A			; zero to A
	ld (iobyte),A		; clear the iobyte
	ld (userdrv),A		; select user disk
	ld HL,mnttab
	ld (0FFFEh),HL
	jp gocpm		; finish up cold boot

; 
print:	ld A,(HL)		; print a zero terinated 
	or A			; string to the console
	ret z
	ld C,A
	call conout
	inc HL
	jp print

; setting alv addresses. 
alv01:	equ alv00 + 255		; allocation vector 1
alv02:	equ alv01 + 255		; allocation vector 2

top:	equ alv02 + 255		; end of BIOS
;
org top
db 00
	END			;of BIOS
